/*
** oidentd_util.h - oidentd utility functions.
** Copyright (c) 1998-2006 Ryan McCabe <ryan@numb.org>
** Copyright (c) 2018      Janik Rabe  <oidentd@janikrabe.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#ifndef __OIDENTD_UTIL_H
#define __OIDENTD_UTIL_H

#include <stdlib.h> /* inline randval() */

#define FACILITY	LOG_DAEMON
#define NORMAL		LOG_INFO
#define DEBUG		LOG_DEBUG

#ifndef MIN
#	define MIN(x,y) ((x) < (y) ? (x) : (y))
#endif

typedef struct list {
	struct list *next;
	void *data;
} list_t;

struct udb_lookup_res {
	/*
	** 0 if no match was found or an error occurred
	** 1 if a local match was found
	** 2 if a non-local match was found and the reply sent to the client
	*/
	char status;

	/*
	** The matching UID if a local match was found
	** Otherwise, MISSING_UID as reserved by POSIX
	*/
	uid_t uid;
};

int o_log(int priority, const char *fmt, ...) __format((printf, 2, 3));
int drop_privs(uid_t new_uid, gid_t new_gid);
int go_background(void);

struct udb_lookup_res get_udb_user(	in_port_t lport,
					in_port_t fport,
					const struct sockaddr_storage *laddr,
					const struct sockaddr_storage *faddr,
					int sock);

FILE *safe_open(const struct passwd *pw, const char *filename);

void *xmalloc(size_t size);
void *xcalloc(size_t nmemb, size_t size);
void *xrealloc(void *ptr, size_t len);

char *xstrncpy(char *dest, const char *src, size_t n);
char *xstrdup(const char *string);

list_t *list_prepend(list_t **list, void *new_data);
void list_destroy(list_t *list, void (*free_data)(void *));

int find_user(const char *temp_user, uid_t *uid);
int find_group(const char *temp_group, gid_t *gid);

int random_seed(void);

/*
** Return a pseudo-random integer between 0 and i-1 inclusive. The
** obvious solution (rand()%i) isn't safe because on many systems,
** the low-order bits of the return of rand()(3) are grossly non-random.
** (FIXME: See comment preceding random_seed() regarding using better
** PRNG functions on systems whose libraries provide them.)
*/

static inline int randval(int i) {
	/* Per _Numerical Recipes in C_: */
	return (int) ((double) i * rand() / (RAND_MAX+1.0));
}

#ifndef HAVE_SNPRINTF
	int snprintf(char *str, size_t n, char const *fmt, ...);
#endif

#ifndef HAVE_VSNPRINTF
	int vsnprintf(char *str, size_t n, char *fmt, va_list ap);
#endif

#endif
